﻿const listEl = document.getElementById('blacklist');
const regInput = document.getElementById('regRules');
const statCountEl = document.getElementById('statCount');

// --- 初始化与列表渲染（合并后的增强版） ---
async function updateUI() {
    const { regRules = "" } = await chrome.storage.local.get(['regRules']);
    regInput.value = regRules;

    listEl.innerHTML = "";
    // 强化过滤：去除空行和首尾空格
    const lines = regRules.split('\n')
        .map(l => l.trim())
        .filter(l => l.length > 0);

    // 过滤出精准作者名用于计数
    const authorList = lines.filter(l => l.startsWith('author:'));
    statCountEl.innerText = authorList.length;

    lines.forEach((line) => {
        const item = document.createElement('div');
        item.className = 'list-item';
        // 根据前缀显示不同的图标和文字
        const displayText = line.startsWith('author:') ?
            `👤 作者: ${line.replace('author:', '')}` :
            `🔍 正则: ${line}`;

        item.innerHTML = `<span>${displayText}</span> <button class="del-btn" data-line="${line}">移除</button>`;
        listEl.appendChild(item);
    });
}

// --- 保存修改 ---
document.getElementById('save').onclick = async () => {
    await chrome.storage.local.set({ regRules: regInput.value });
    alert('保存成功！刷新B站主页生效。');
    updateUI();
};

// --- 删除单条规则 ---
listEl.onclick = async (e) => {
    if (e.target.classList.contains('del-btn')) {
        const lineToRemove = e.target.dataset.line;
        const { regRules = "" } = await chrome.storage.local.get(['regRules']);
        const newRules = regRules.split('\n')
            .filter(l => l.trim() !== lineToRemove)
            .join('\n');
        await chrome.storage.local.set({ regRules: newRules });
        updateUI();
    }
};

// --- 导出功能 ---
document.getElementById('exportBtn').onclick = async () => {
    const data = await chrome.storage.local.get(['regRules']);
    const blob = new Blob([JSON.stringify(data, null, 4)], { type: 'application/json' });
    const url = URL.createObjectURL(blob);
    const a = document.createElement('a');
    a.href = url;
    a.download = `Bilibili_Blacklist_Backup_${new Date().toLocaleDateString()}.json`;
    a.click();
    URL.revokeObjectURL(url);
};

// --- 导入功能 ---
const fileInput = document.getElementById('fileInput');
document.getElementById('importBtn').onclick = () => fileInput.click();

fileInput.onchange = (e) => {
    const file = e.target.files[0];
    if (!file) return;
    const reader = new FileReader();
    reader.onload = async (event) => {
        try {
            const importedData = JSON.parse(event.target.result);
            if (importedData.regRules !== undefined) {
                if (confirm('导入将覆盖现有的屏蔽列表，确定吗？')) {
                    await chrome.storage.local.set({ regRules: importedData.regRules });
                    alert('导入成功！');
                    updateUI();
                }
            } else { alert('无效的备份文件'); }
        } catch (err) { alert('读取失败，请检查文件格式'); }
    };
    reader.readAsText(file);
};

// --- 安全清空功能（先导出再清空） ---
document.getElementById('safeClearBtn').onclick = async () => {
    const { regRules = "" } = await chrome.storage.local.get(['regRules']);

    if (!regRules || regRules.trim() === "") {
        alert("列表已经是空的，无需清理。");
        return;
    }

    if (confirm("确定要清空所有规则吗？\n系统将自动先下载一份备份文件，下载完成后将立即执行清空。")) {
        try {
            // 1. 执行备份导出
            const data = { regRules: regRules };
            const blob = new Blob([JSON.stringify(data, null, 4)], { type: 'application/json' });
            const url = URL.createObjectURL(blob);
            const a = document.createElement('a');
            a.href = url;
            a.download = `Auto_Backup_Before_Clear_${new Date().getTime()}.json`;
            document.body.appendChild(a);
            a.click();
            document.body.removeChild(a);
            URL.revokeObjectURL(url);

            // 2. 延时清空存储
            setTimeout(async () => {
                await chrome.storage.local.set({ regRules: "" });
                alert("已成功导出备份并清空所有规则。");
                updateUI();
            }, 500);
        } catch (err) {
            alert("操作失败：" + err.message);
        }
    }
};

// 初始加载
updateUI();