﻿(async function () {
    console.log("【B站黑名单插件】V5.0：占位折叠模式（彻底修复虚拟列表死循环）");

    let { regRules = "" } = await chrome.storage.local.get(['regRules']);

    // 穿透 Shadow DOM
    function querySelectorAllDeep(selector, root = document) {
        let nodes = Array.from(root.querySelectorAll(selector));
        const walker = document.createTreeWalker(root, NodeFilter.SHOW_ELEMENT, null, false);
        let node = walker.nextNode();
        while (node) {
            if (node.shadowRoot) {
                nodes = nodes.concat(querySelectorAllDeep(selector, node.shadowRoot));
            }
            node = walker.nextNode();
        }
        return nodes;
    }

    // 规则解析
    function getRuleSets(rules) {
        if (!rules) return { authorRegex: null, generalRegex: null };
        const lines = rules.split('\n').map(l => l.trim()).filter(l => l.length > 0);
        const authorLines = lines.filter(l => l.startsWith('author:')).map(l => l.replace('author:', ''));
        const generalLines = lines.filter(l => !l.startsWith('author:'));

        return {
            authorRegex: authorLines.length ? new RegExp(`^(${authorLines.join('|')})$`, 'i') : null,
            generalRegex: generalLines.length ? new RegExp(generalLines.join('|'), 'i') : null
        };
    }

    let ruleSets = getRuleSets(regRules);

    /**
     * 【核心修复】占位折叠函数
     * 不再使用 display:none，而是将卡片内容清空，变成一个很小的占位块
     */
    function collapseCard(card, reason) {
        // 如果已经处理过，直接跳过（防止重复操作导致闪烁）
        if (card.getAttribute('data-is-blocked') === 'true') return;

        // 1. 标记为已屏蔽
        card.setAttribute('data-is-blocked', 'true');

        // 2. 锁定原有样式，防止 Bewly 覆盖
        // 我们保留 display: flex/block 等布局属性，但强制缩小尺寸
        card.style.setProperty('visibility', 'visible', 'important'); // 必须可见，否则虚拟列表会以为它不存在
        card.style.setProperty('min-height', '0', 'important');
        card.style.setProperty('height', '30px', 'important'); // 保留 30px 高度作为“稳定锚点”
        card.style.setProperty('overflow', 'hidden', 'important');
        card.style.setProperty('opacity', '0.6', 'important');
        card.style.setProperty('background', '#f1f2f3', 'important');
        card.style.setProperty('border-radius', '4px', 'important');
        card.style.setProperty('margin-bottom', '4px', 'important');
        card.style.setProperty('cursor', 'not-allowed', 'important');
        card.style.setProperty('pointer-events', 'none', 'important');

        // 3. 替换内部 HTML 为简单的提示文本 (可选，不想看文字可以把 innerHTML 设为空)
        // 这样做的好处是大大减少渲染压力，浏览器不用渲染封面图和视频流
        card.innerHTML = `<div style="
            text-align: center; 
            line-height: 30px; 
            font-size: 12px; 
            color: #999;
            font-family: sans-serif;
        ">🚫 已屏蔽规则匹配内容</div>`;
    }

    function processCard(card) {
        // 如果卡片已经被折叠过，就不需要再读取内容了（节省性能）
        if (card.getAttribute('data-is-blocked') === 'true') return;

        const authorEl = card.querySelector('.channel-name');
        const titleEl = card.querySelector('.video-card-title');
        // Bewly 有时会渲染骨架屏（Skeleton），没有内容时跳过
        if (!authorEl || !titleEl) return;

        const authorName = authorEl.innerText.trim();
        const videoTitle = titleEl.innerText.trim();

        let shouldBlock = false;

        if (ruleSets.authorRegex && ruleSets.authorRegex.test(authorName)) shouldBlock = true;
        if (!shouldBlock && ruleSets.generalRegex) {
            if (ruleSets.generalRegex.test(authorName) || ruleSets.generalRegex.test(videoTitle)) shouldBlock = true;
        }

        if (shouldBlock) {
            collapseCard(card);
        } else {
            // 如果没屏蔽，确保有按钮
            if (!card.querySelector('.my-block-btn')) {
                injectBtn(card, authorEl);
            }
        }
    }

    function injectBtn(card, authorEl) {
        if (card.querySelector('.my-block-btn')) return;

        const btn = document.createElement('span');
        btn.innerHTML = '🚫';
        btn.className = 'my-block-btn';
        btn.style.cssText = 'cursor:pointer !important; color:#ff4d4f !important; font-weight:bold; margin-right:8px; font-size:16px; display:inline-block; flex-shrink:0; position:relative; z-index:999 !important; pointer-events: auto !important;';

        btn.addEventListener('click', async (e) => {
            e.preventDefault();
            e.stopPropagation();

            // 重新抓取名字
            const currentAuthorEl = card.querySelector('.channel-name');
            if (!currentAuthorEl) return;
            const currentAuthorName = currentAuthorEl.innerText.trim();

            let data = await chrome.storage.local.get('regRules');
            let currentRules = data.regRules || "";
            const taggedAuthor = `author:${currentAuthorName}`;

            if (!currentRules.includes(taggedAuthor)) {
                const newRules = currentRules.trim() + (currentRules ? "\n" : "") + taggedAuthor;
                await chrome.storage.local.set({ regRules: newRules });
            }

            // 点击后立即折叠
            collapseCard(card);
        }, true);

        authorEl.prepend(btn);
    }

    // 优化：使用 MutationObserver 代替 setInterval
    // 虚拟列表更新非常快，Observer 能更精准地捕获变化，减少 CPU 空转
    const observer = new MutationObserver((mutations) => {
        // 简单防抖，避免一帧内触发太多次
        requestAnimationFrame(() => {
            querySelectorAllDeep('.video-card-container').forEach(processCard);
        });
    });

    // 开始监听整个 body，配置为监听子树变化
    observer.observe(document.body, { childList: true, subtree: true });

    // 兜底轮询：防止 Observer 漏掉某些 Shadow DOM 的变化
    setInterval(() => {
        querySelectorAllDeep('.video-card-container').forEach(processCard);
    }, 1500);

    chrome.storage.onChanged.addListener((changes) => {
        if (changes.regRules) {
            ruleSets = getRuleSets(changes.regRules.newValue);
            // 规则更新后，强制重新扫描所有卡片（包括没折叠的）
            // 注意：已折叠的没法恢复，除非刷新页面，这是 DOM 替换的代价，但保证了稳定
            querySelectorAllDeep('.video-card-container').forEach(c => c.removeAttribute('data-is-blocked'));
        }
    });
})();