﻿// 屏蔽作者功能
let blockedAuthors = new Set();

// 初始化
function initializeShieldButtons() {
    loadBlockedAuthors();
    addShieldButtons();
    startObserving();
}

// 加载已屏蔽的作者
function loadBlockedAuthors() {
    chrome.storage.local.get(['blockedAuthors'], function (result) {
        if (result.blockedAuthors) {
            blockedAuthors = new Set(result.blockedAuthors);
            console.log('已加载屏蔽作者:', Array.from(blockedAuthors));
        }
    });
}

// 保存屏蔽的作者
function saveBlockedAuthors() {
    const authorsArray = Array.from(blockedAuthors);
    chrome.storage.local.set({ blockedAuthors: authorsArray });
}

// 添加屏蔽按钮到视频卡片
function addShieldButtons() {
    const videoCards = document.querySelectorAll('.bili-video-card, .video-card, [class*="video"]');

    videoCards.forEach(card => {
        // 跳过已经隐藏的卡片
        if (card.style.display === 'none') return;

        const authorElement = findAuthorElement(card);
        if (authorElement && !authorElement.querySelector('.bili-shield-btn')) {
            addButtonToAuthor(authorElement, card);
        }
    });
}

// 查找作者元素
function findAuthorElement(card) {
    const authorSelectors = [
        '.bili-video-card__info--author',
        '.up-name',
        '.bili-video-card__info--author--name',
        '.author',
        '.bili-video-card__info--owner',
        '.bili-video-card__info--bottom'
    ];

    for (let selector of authorSelectors) {
        const element = card.querySelector(selector);
        if (element) return element;
    }
    return null;
}

// 新的按钮插入逻辑 - 按钮在前，作者名在后
function addButtonToAuthor(authorElement, card) {
    const originalText = authorElement.textContent.trim();
    const authorName = originalText.replace(/屏蔽作者|已屏蔽/g, '').trim();

    // 如果已经处理过这个元素，跳过
    if (authorElement.classList.contains('has-shield-btn')) {
        return;
    }

    // 标记为已处理
    authorElement.classList.add('has-shield-btn');

    try {
        // 新的方法：创建容器，按钮在前，作者名在后
        createButtonFirstLayout(authorElement, authorName, card);
    } catch (error) {
        console.log('主方法失败，尝试备用方法:', error);
        try {
            // 备用方法：直接在元素前插入按钮
            addButtonBeforeAuthor(authorElement, authorName, card);
        } catch (error2) {
            console.log('备用方法失败，尝试底部添加:', error2);
            // 最后方法：在卡片底部添加
            addButtonToCardBottom(card, authorName);
        }
    }
}

// 主方法：创建按钮在前的布局
function createButtonFirstLayout(authorElement, authorName, card) {
    // 保存原始内容
    const originalContent = authorElement.innerHTML;

    // 创建容器
    const container = document.createElement('div');
    container.className = 'author-container';

    // 创建屏蔽按钮
    const shieldBtn = createShieldButton(authorName, card);

    // 创建作者名显示区域
    const nameSpan = document.createElement('span');
    nameSpan.className = 'author-name';
    nameSpan.textContent = authorName;
    nameSpan.title = authorName; // 添加悬停提示

    // 组装：按钮在前，作者名在后
    container.appendChild(shieldBtn);
    container.appendChild(nameSpan);

    // 替换原内容
    authorElement.innerHTML = '';
    authorElement.appendChild(container);
}

// 备用方法：在作者元素前插入按钮
function addButtonBeforeAuthor(authorElement, authorName, card) {
    // 创建屏蔽按钮
    const shieldBtn = createShieldButton(authorName, card);

    // 在作者元素前插入按钮
    authorElement.parentNode.insertBefore(shieldBtn, authorElement);

    // 确保作者名不会被挤到下一行
    authorElement.style.display = 'inline';
    authorElement.style.maxWidth = 'calc(100% - 70px)'; // 预留按钮空间
    authorElement.style.overflow = 'hidden';
    authorElement.style.textOverflow = 'ellipsis';
    authorElement.style.whiteSpace = 'nowrap';
}

// 创建屏蔽按钮的通用函数
function createShieldButton(authorName, card) {
    const shieldBtn = document.createElement('button');
    shieldBtn.className = 'bili-shield-btn';
    shieldBtn.innerHTML = '屏蔽作者';
    shieldBtn.title = '不再显示此作者的内容';

    // 如果已经屏蔽，更新按钮状态
    if (blockedAuthors.has(authorName)) {
        shieldBtn.classList.add('blocked');
        shieldBtn.innerHTML = '已屏蔽';
        shieldBtn.disabled = true;
    }

    // 点击事件
    shieldBtn.addEventListener('click', function (e) {
        e.preventDefault();
        e.stopPropagation();
        blockAuthor(authorName, card, shieldBtn);
    });

    return shieldBtn;
}

// 在卡片底部添加按钮（最后备选）
function addButtonToCardBottom(card, authorName) {
    // 查找合适的插入位置
    const insertSelectors = [
        '.bili-video-card__info--bottom',
        '.bili-video-card__info',
        '.video-card__info',
        '.bili-video-card__content'
    ];

    let insertElement = card;
    for (let selector of insertSelectors) {
        const element = card.querySelector(selector);
        if (element) {
            insertElement = element;
            break;
        }
    }

    // 创建屏蔽按钮
    const shieldBtn = document.createElement('button');
    shieldBtn.className = 'bili-shield-btn';
    shieldBtn.innerHTML = `屏蔽作者: ${authorName.substring(0, 8)}${authorName.length > 8 ? '...' : ''}`;
    shieldBtn.title = `不再显示作者"${authorName}"的内容`;
    shieldBtn.style.marginTop = '4px';
    shieldBtn.style.marginBottom = '4px';
    shieldBtn.style.width = 'auto';

    // 如果已经屏蔽，更新按钮状态
    if (blockedAuthors.has(authorName)) {
        shieldBtn.classList.add('blocked');
        shieldBtn.innerHTML = `已屏蔽: ${authorName.substring(0, 8)}${authorName.length > 8 ? '...' : ''}`;
        shieldBtn.disabled = true;
    }

    // 点击事件
    shieldBtn.addEventListener('click', function (e) {
        e.preventDefault();
        e.stopPropagation();
        blockAuthor(authorName, card, shieldBtn);
    });

    // 插入到元素末尾
    insertElement.appendChild(shieldBtn);
}

// 屏蔽作者
function blockAuthor(authorName, card, button) {
    if (blockedAuthors.has(authorName)) return;

    // 添加到屏蔽列表
    blockedAuthors.add(authorName);
    saveBlockedAuthors();

    // 更新按钮状态
    button.classList.add('blocked');
    button.innerHTML = '已屏蔽';
    button.disabled = true;

    // 隐藏该视频卡片
    card.style.display = 'none';

    // 显示确认提示
    showShieldConfirm(authorName);

    console.log(`已屏蔽作者: ${authorName}`);
}

// 显示屏蔽确认提示
function showShieldConfirm(authorName) {
    // 移除已有的提示
    const existingToast = document.querySelector('.shield-confirm-toast');
    if (existingToast) {
        existingToast.remove();
    }

    // 创建新提示
    const toast = document.createElement('div');
    toast.className = 'shield-confirm-toast';
    toast.innerHTML = `✅ 已屏蔽作者: ${authorName}`;

    document.body.appendChild(toast);

    // 3秒后自动消失
    setTimeout(() => {
        if (toast.parentNode) {
            toast.style.animation = 'slideIn 0.3s ease reverse';
            setTimeout(() => toast.remove(), 300);
        }
    }, 3000);
}

// 更新检查视频卡片的函数
function checkVideoCards() {
    chrome.storage.local.get(['keywords', 'blockedAuthors'], function (result) {
        const keywords = result.keywords || [];
        const blockedAuthorsList = result.blockedAuthors || [];

        const videoCards = document.querySelectorAll('.bili-video-card, .video-card, [class*="video"]');

        videoCards.forEach(card => {
            // 跳过已经隐藏的卡片
            if (card.style.display === 'none') return;

            const titleElement = findTitleElement(card);
            const authorElement = findAuthorElement(card);

            if (titleElement && authorElement) {
                const title = titleElement.textContent;
                // 从容器中提取纯作者名
                const authorNameSpan = authorElement.querySelector('.author-name');
                const authorText = authorNameSpan ?
                    authorNameSpan.textContent :
                    authorElement.textContent.replace(/屏蔽作者|已屏蔽/g, '').trim();

                // 检查关键词屏蔽
                const keywordMatch = keywords.some(item => {
                    try {
                        if (item.isRegex) {
                            const regex = new RegExp(item.keyword, 'i');
                            return regex.test(title) || regex.test(authorText);
                        } else {
                            return title.includes(item.keyword) || authorText.includes(item.keyword);
                        }
                    } catch (e) {
                        console.error('正则表达式错误:', e);
                        return false;
                    }
                });

                // 检查作者屏蔽
                const authorMatch = blockedAuthorsList.includes(authorText);

                if (keywordMatch || authorMatch) {
                    card.style.display = 'none';
                }
            }
        });

        // 确保所有作者都有屏蔽按钮
        addShieldButtons();
    });
}

// 查找标题元素
function findTitleElement(card) {
    const titleSelectors = [
        '.bili-video-card__info--tit',
        '.title',
        '.video-title',
        '.bili-video-card__info--title',
        'h3',
        'h4'
    ];

    for (let selector of titleSelectors) {
        const element = card.querySelector(selector);
        if (element) return element;
    }
    return null;
}

// 启动监控
function startObserving() {
    const observer = new MutationObserver(function (mutations) {
        let shouldCheck = false;

        mutations.forEach(mutation => {
            if (mutation.addedNodes.length > 0) {
                shouldCheck = true;
            }
        });

        if (shouldCheck) {
            setTimeout(() => {
                checkVideoCards();
                addShieldButtons();
            }, 500);
        }
    });

    observer.observe(document.body, {
        childList: true,
        subtree: true
    });

    // 初始检查
    setTimeout(() => {
        checkVideoCards();
        addShieldButtons();
    }, 2000);
}

// 监听来自popup的消息
chrome.runtime.onMessage.addListener(function (request, sender, sendResponse) {
    if (request.action === 'updateBlockedAuthors') {
        blockedAuthors = new Set(request.authors);
        // 重新检查所有卡片状态
        checkVideoCards();
    }
});

// 页面加载完成后初始化
if (document.readyState === 'loading') {
    document.addEventListener('DOMContentLoaded', initializeShieldButtons);
} else {
    initializeShieldButtons();
}